﻿//
// ChibiMo3 Sample App
// Copyright (C) 2013  kuroi kouichi @ q61.org <ko@q61.org>
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using System.Drawing.Imaging;
using System.Runtime.InteropServices;

namespace WindowsFormsApplication1
{
    class ChibimoImageBuffer
    {
        public enum LCDType
        {
            LS027B4DH01,
            KS0108,
            ST7565R
        };

        public ChibimoImageBuffer(LCDType lcdtype, int width, int height)
        {
            _lcdtype = lcdtype;
            _width = width;
            _height = height;

            // validate parameters
            switch (_lcdtype) {
                case LCDType.LS027B4DH01:
                    if (_width != 400) {
                        throw new ArgumentOutOfRangeException("width");
                    }
                    if (_height != 240) {
                        throw new ArgumentOutOfRangeException("height");
                    }
                    break;
                default:
                    throw new ArgumentOutOfRangeException("lcdtype");
            }

            _data = new byte[_width * _height];
        }

        public void Clear()
        {
            for (int i = 0; i < _data.Length; i++) {
                _data[i] = 240;
            }
        }

        public void SetDataAtPixel(byte data, int x, int y)
        {
            if ((x < 0) || (x >= _width)) return;
            if ((y < 0) || (y >= _height)) return;

            _SetDataAtPixel(data, x, y);
        }

        public void SetDataFromImage(Image srcimg)
        {
            // alloc intermediate bitmap and get data pointer
            Bitmap bmp = new Bitmap(srcimg);
            int srcwd = bmp.Width;
            int srcht = bmp.Height;
            int srccenterx = srcwd / 2;
            int srccentery = srcht / 2;

            BitmapData bmpdt = bmp.LockBits(new Rectangle(0, 0, srcwd, srcht),
                ImageLockMode.ReadOnly, PixelFormat.Format24bppRgb);

            // clear current buffer
            Clear();

            // scan src image and get data at center
            try {
                byte[] dtstride = new byte[bmpdt.Stride];
                for (int yi = 0; yi < _height; yi++) {
                    int y = srccentery - _height / 2 + yi;
                    if (y < 0) continue;
                    if (y >= srcht) break;

                    Marshal.Copy(bmpdt.Scan0 + (bmpdt.Stride * y), dtstride, 0, bmpdt.Stride);
                    for (int xi = 0; xi < _width; xi++) {
                        int x = srccenterx - _width / 2 + xi;
                        if (x < 0) continue;
                        if (x >= srcwd) break;

                        int graydata = 0;
                        graydata += dtstride[x * 3 + 2] * 66; // red
                        graydata += dtstride[x * 3 + 1] * 129; // green
                        graydata += dtstride[x * 3 + 0] * 25; // blue
                        _SetDataAtPixel((byte)(graydata >> 8), xi, yi);
                    }
                }
            } finally {
                bmp.UnlockBits(bmpdt);
            }
        }
            
        public void ApplyFloydSteinberg()
        {
            for (int y = 0; y < _height; y++) {
                for (int x = 0; x < _width; x++) {
                    int oldpx = _DataAtPixel(x, y);
                    int newpx = (oldpx < 128) ? 16 : 240;
                    int qerr = oldpx - newpx;

                    _SetDataAtPixel((byte)newpx, x, y);
                    if (x < _width - 1) {
                        _AddDataAtPixel(qerr * 7 / 16, x + 1, y);
                    }
                    if (y < _height - 1) {
                        if (x > 0) {
                            _AddDataAtPixel(qerr * 3 / 16, x - 1, y + 1);
                        }
                        _AddDataAtPixel(qerr * 5 / 16, x, y + 1);
                        if (x < _width - 1) {
                            _AddDataAtPixel(qerr * 1 / 16, x + 1, y + 1);
                        }
                    }
                }
            }
        }

        public Image CreateImageFromContent()
        {
            Bitmap rtbmp = new Bitmap(_width, _height);
            for (int y = 0; y < _height; y++) {
                for (int x = 0; x < _width; x++) {
                    rtbmp.SetPixel(x, y, Color.FromKnownColor(
                        (_DataAtPixel(x, y) < 128 ? KnownColor.Black : KnownColor.White)));
                }
            }

            return (rtbmp);
        }

        public byte[] CreateChibimoCommandData()
        {
            switch (_lcdtype) {
                case LCDType.LS027B4DH01: {
                        byte[] rt = new byte[(8 + _width / 8) * _height];
                        int pos = 0;

                        for (int y = 0; y < _height; y++) {
                            // line start command
                            rt[pos++] = 0x80;
                            rt[pos++] = 0x80;
                            // line number
                            rt[pos++] = (byte)(y + 1);

                            // pixel data
                            for (int xi = 0; xi < _width / 8; xi++) {
                                byte dt = 0;
                                for (int k = 0; k < 8; k++) {
                                    dt >>= 1;
                                    dt |= (byte)((_DataAtPixel(xi * 8 + k, y) < 128) ? 0 : 0x80);
                                }
                                rt[pos++] = dt;
                            }

                            // trailing command
                            rt[pos++] = 0;
                            rt[pos++] = 0;
                            rt[pos++] = 0;
                            rt[pos++] = 0;
                            rt[pos++] = 0;
                        }
                        return (rt);
                    }

                default: // other LCDs not supported yet
                    return (null);
            }
        }

        public string CreateCArrayRepresentation()
        {
            string rt;

            switch (_lcdtype) {
                case LCDType.LS027B4DH01:
                    rt = string.Format("unsigned char  chibimodata[{0}] = ",
                        (_width / 8 + 4) * _height);

                    rt += "{";
                    for (int y = 0; y < _height; y++) {
                        rt += "\r\n";

                        // scanline update command
                        rt += " 0x01,";
                        // line number
                        rt += string.Format(" 0x{0:x2},", y + 1);

                        // pixel data
                        for (int xi = 0; xi < _width / 8; xi++) {
                            byte dt = 0;
                            for (int k = 0; k < 8; k++) {
                                dt >>= 1;
                                dt |= (byte)((_DataAtPixel(xi * 8 + k, y) < 128) ? 0 : 0x80);
                            }

                            rt += string.Format(" 0x{0:x2},", dt);
                        }

                        // trail data
                        rt += " 0x00, 0x00,";
                    }

                    rt = rt.Remove(rt.Length - 1);
                    rt += "\r\n};\r\n";
                    break;
                default: // other LCDs not supported yet
                    return (null);
            }

            return (rt);
        }


        public int Width { get { return (_width); } }
        public int Height { get { return (_height); } }

        private byte _DataAtPixel(int x, int y)
        {
            return (_data[_width * y + x]);
        }

        private void _SetDataAtPixel(byte dt, int x, int y)
        {
            _data[_width * y + x] = dt;
        }

        private void _AddDataAtPixel(int dt, int x, int y)
        {
            int curdt = _DataAtPixel(x, y);
            curdt += dt;
            if (curdt < 0) curdt = 0;
            if (curdt > 255) curdt = 255;
            _SetDataAtPixel((byte)curdt, x, y);
        }

        private byte[] _data;
        private LCDType _lcdtype;
        private int _width, _height;
    }
}
